package main

import (
	"github.com/veandco/go-sdl2/img"
	"github.com/veandco/go-sdl2/sdl"
)

type Asteroid struct {
	startX float32
	startY float32
	x      float32
	y      float32
	angle  float32
	speed  float32
	sprite *Sprite
	resets int
	dead   bool
}

var asteroidAsset []byte
var asteroidRWops *sdl.RWops
var asteroidSurface *sdl.Surface

func loadAsteroid() *sdl.Surface {
	if asteroidSurface == nil {
		var err error
		asteroidAsset, err = Asset("res/asteroid.png")
		if err != nil {
			panic(err)
		}
		asteroidRWops, err = sdl.RWFromMem(asteroidAsset)
		if err != nil {
			panic(err)
		}
		asteroidSurface, err = img.LoadPNGRW(asteroidRWops)
		if err != nil {
			panic(err)
		}
	}
	return asteroidSurface
}

func NewAsteroid(renderer *sdl.Renderer, x, y, speed, angle float32) *Asteroid {

	asteroid := new(Asteroid)
	asteroid.sprite = NewSprite(renderer, loadAsteroid())
	asteroid.startX = x
	asteroid.startY = y
	asteroid.x = x
	asteroid.y = y
	asteroid.speed = speed
	asteroid.resets = 0
	asteroid.dead = false
	asteroid.angle = angle
	return asteroid
}

func (a *Asteroid) Update(deltaTime float32) {
	a.y += a.speed * deltaTime
}

func (a *Asteroid) Draw(renderer *sdl.Renderer) {
	a.sprite.RotateAndDrawCenter(renderer, float64(a.angle), int32(a.x), int32(a.y))
}

type AsteroidField struct {
	asteroids []*Asteroid
	allDead   bool
	onDead    func()
}

func NewAsteroidField(renderer *sdl.Renderer, onDead func()) *AsteroidField {
	af := new(AsteroidField)
	af.allDead = false
	af.onDead = onDead
	asteroids := (screenWidth - 264 - 264) / 64
	for i := 0; i < int(asteroids); i++ {
		x := 264 + (64 * i)
		if x < int(screenWidth/2)-int(rymdskeppTextureWidth) || x > int(screenWidth/2)+int(rymdskeppTextureWidth) {
			af.asteroids = append(af.asteroids, NewAsteroid(renderer,
				float32(x),
				float32(RandomNumber(-int(screenHeight), -64)),
				float32(RandomNumber(100, 150))/1000.0,
				float32(RandomNumber(0, 360))))
		}
	}
	return af
}

func (af *AsteroidField) Update(deltaTime float32) {
	if af.allDead {
		return
	}
	for _, asteroid := range af.asteroids {
		if !asteroid.dead {
			if asteroid.y < float32(screenHeight+32) {
				asteroid.Update(deltaTime)
			} else {
				if asteroid.resets > 0 {
					asteroid.x = asteroid.startX
					asteroid.y = asteroid.startY
					asteroid.resets--
				} else {
					asteroid.dead = true
				}
			}
		}
	}
	af.allDead = true
	for _, asteroid := range af.asteroids {
		if !asteroid.dead {
			af.allDead = false
		}
	}
	if af.allDead {
		if af.onDead != nil {
			af.onDead()
		}
	}
}

func (af *AsteroidField) Draw(renderer *sdl.Renderer) {
	for _, asteroid := range af.asteroids {
		if !asteroid.dead {
			if asteroid.y < float32(screenHeight+32) {
				asteroid.Draw(renderer)
			}
		}
	}
}
